﻿using UnityEngine;
using System.Collections;
using System.Collections.Generic;

public class DudeController : MonoBehaviour {

	// Horizontal movement.
	private bool facingRight_ = true;		// For determining which way the player is currently facing.
	private float h_ = 0.0f;				// Horizontal axis raw input.
	private bool moving_ = false;			// Condition for whether the player is moving along the x axis.
	public float MoveForce = 50;			// Amount of force added to move the player left and right.
	public float MaxSpeed = 1.5f;			// The fastest the player can travel in the x axis.
	public float MinSpeed = 0.001f;			// The slowest the player can move along the x axis.

	// Vertical movement.
	private bool jump_ = false;				// Condition for whether the player should jump.
	public float JumpForce = 150;
	private List<GameObject> groundCollisionList_ = new List<GameObject>();
	private float elapsedAirbornTime_ = 0.0f;	// Number of seconds player has been airborn.
	public float MaxJumpLagTime = 0.1f;		// Number of seconds the player is airborn and allowed to jump.

	// Components.
	private Animator anim_;					// Reference to the player's animator component.

	//for hacking around annoying relative velocity springboard bug
	private Vector2 previousVelocity = new Vector2(0,0);

	public Vector2 GetPreviousVelocity() {
		return previousVelocity;
	}
	
	void Awake()
	{
		// Setting up references.
		anim_ = GetComponent<Animator>();
	}

	void Update()
	{
		// Set the grounded state of the animator.
		if (anim_)
			anim_.SetBool("Grounded", groundCollisionList_.Count > 0);

		// If the player is airborn, accumulate the airborn timer.
		if (groundCollisionList_.Count == 0)
			elapsedAirbornTime_ += Time.deltaTime;
		else
			elapsedAirbornTime_ = MaxJumpLagTime;

		// If the jump button is pressed and the player is grounded then the player should jump.
		if(Input.GetButtonDown("Jump") && (groundCollisionList_.Count > 0 || elapsedAirbornTime_ < MaxJumpLagTime))
		{
			jump_ = true;
			elapsedAirbornTime_ = MaxJumpLagTime;

			// Stop the player's y-axis motion.
			GetComponent<Rigidbody2D>().velocity = new Vector2(GetComponent<Rigidbody2D>().velocity.x, 0);
		}
		
		// Cache the horizontal input.
		h_ = Input.GetAxisRaw("Horizontal");
		if (h_ != 0) 
			moving_ = true;
		else 
			moving_ = false;
	}

	void FixedUpdate ()
	{
		// The Speed animator parameter is set to the absolute value of the horizontal velocity
		anim_.SetFloat("Speed", Mathf.Abs(GetComponent<Rigidbody2D>().velocity.x)); 

		// If the player is changing direction (h_ has a different sign to velocity.x) or hasn't reached maxSpeed yet...
		if(h_ * GetComponent<Rigidbody2D>().velocity.x < MaxSpeed)
		{
			// ... add a force to the player.
			Vector2 force = Vector2.right * h_ * MoveForce;
			GetComponent<Rigidbody2D>().AddForce(force);
		}

		// If the player's horizontal velocity is greater than the maxSpeed...
		if(Mathf.Abs(GetComponent<Rigidbody2D>().velocity.x) > MaxSpeed)
			// ... set the player's velocity to the maxSpeed in the x axis.
			GetComponent<Rigidbody2D>().velocity = new Vector2(Mathf.Sign(GetComponent<Rigidbody2D>().velocity.x) * MaxSpeed, GetComponent<Rigidbody2D>().velocity.y);

		// If the input is moving the player right and the player is facing left...
		if(h_ > 0 && !facingRight_)
			// ... flip the player.
			Flip();
		// Otherwise if the input is moving the player left and the player is facing right...
		else if(h_ < 0 && facingRight_)
			// ... flip the player.
			Flip();

		// If the player should jump...
		if(jump_)
		{
			// Add a vertical force to the player.
			GetComponent<Rigidbody2D>().AddForce(new Vector2(0f, JumpForce));

			// Make sure the player can't jump again until the jump conditions from Update are satisfied.
			jump_ = false;
		}

		// If the player just stopped moving, apply opposing force to slow down.
		if (!moving_ && Mathf.Abs(GetComponent<Rigidbody2D>().velocity.x) > 0.0f) 
		{
			GetComponent<Rigidbody2D>().velocity = new Vector2(GetComponent<Rigidbody2D>().velocity.x, GetComponent<Rigidbody2D>().velocity.y);
		}

		//springboard hack
		previousVelocity = GetComponent<Rigidbody2D>().velocity;
	}

	void Flip ()
	{
		// Switch the way the player is labelled as facing.
		facingRight_ = !facingRight_;

		// Multiply the player's x local scale by -1.
		Vector3 theScale = transform.localScale;
		theScale.x *= -1;
		transform.localScale = theScale;

		moving_ = true;
	}

	void OnCollisionEnter2D(Collision2D coll) {
		// If the collision is with the ground, track it.
		if (IsGroundCollision(coll)) {
			if (!groundCollisionList_.Contains(coll.gameObject))
				groundCollisionList_.Add(coll.gameObject);
		}
	}

	void OnCollisionExit2D(Collision2D coll) {
		bool grounded = groundCollisionList_.Count > 0;

		// Remove the collision if it's in the ground collision list.
		if (groundCollisionList_.Contains(coll.gameObject))
			groundCollisionList_.Remove(coll.gameObject);

		// If the player was on the ground and is no longer, 
		// reset the elapsed airborn timer.
		if (grounded && groundCollisionList_.Count == 0)
			elapsedAirbornTime_ = 0.0f;
	}

	private bool IsGroundCollision(Collision2D coll) {
		return (coll != null && 
				(coll.gameObject.layer == LayerMask.NameToLayer("Ground") || 
				 coll.gameObject.layer == LayerMask.NameToLayer("Fire")) &&
				coll.contacts[0].normal.y >= 0.95f &&
				coll.gameObject.layer != LayerMask.NameToLayer("Springboard"));
	}
}
