using UnityEngine;

/**
 * A camera to help with Orthagonal mode when you need it to lock to pixels.  Desiged to be used on android and retina devices.
 * 
 * Written by pixelrevision, found here: https://gist.github.com/pixelrevision/2981863
 * 
 * Modified by me! Jonathan Rust
 */
public class PixelPerfectCam : MonoBehaviour {
	public Vector2 targetViewportSizeInPixels = 
		new Vector2(480.0f, 320.0f); //target size of viewport
	public bool lockToPixels = true; //snap movement of camera to pixels
	public float pixelsPerUnit = 32.0f; //number of pixels per Unity unit
	public GameObject followTarget; //target for camera to follow
	public Bounds levelBounds; //bounds beyond which camera won't look
	
	private Camera _camera;
	private int _currentScreenWidth = 0;
	private int _currentScreenHeight = 0;
	
	private float _pixelLockedPPU = 100.0f;
	private Vector2 _winSize;
	
	protected void Start(){
		_camera = this.GetComponent<Camera>();
		if(!_camera){
			Debug.LogWarning("No camera for pixel perfect cam to use");
		}else{
			_camera.orthographic = true;
			ResizeCamToTargetSize();
		}
	}
	
	public void ResizeCamToTargetSize(){
		if(_currentScreenWidth != Screen.width || _currentScreenHeight != Screen.height){
			// check our target size here to see how much we want to scale this camera
			float percentageX = Screen.width/targetViewportSizeInPixels.x;
			float percentageY = Screen.height/targetViewportSizeInPixels.y;
			float targetSize = 0.0f;
			if(percentageX > percentageY){
				targetSize = percentageY;
			}else{
				targetSize = percentageX;
			}
			int floored = Mathf.FloorToInt(targetSize);
			if(floored < 1){
				floored = 1;
			}
			// now we have our percentage let's make the viewport scale to that
			float camSize = (Screen.height/2)/pixelsPerUnit;
			_camera.orthographicSize = camSize;
			_pixelLockedPPU = floored * pixelsPerUnit;
		}
		_winSize = new Vector2(Screen.width, Screen.height);
	}
	
	public void Update(){
		if(_winSize.x != Screen.width || _winSize.y != Screen.height){
			ResizeCamToTargetSize();
		}
		if(_camera && followTarget){
			//get target position
			Vector2 newPosition = new Vector2(followTarget.transform.position.x, followTarget.transform.position.y);
			
			//clamp to bounds
			//clamp min x
			if (newPosition.x < levelBounds.min.x + _winSize.x/_pixelLockedPPU/2.0f) {
				newPosition.x = levelBounds.min.x + _winSize.x/_pixelLockedPPU/2.0f;
			}
			//clamp max x
			else if (newPosition.x > levelBounds.max.x - _winSize.x/_pixelLockedPPU/2.0f) {
				newPosition.x = levelBounds.max.x - _winSize.x/_pixelLockedPPU/2.0f;
			}
			//center x if level width is smaller than camera width
			if (Mathf.Abs(levelBounds.max.x - levelBounds.min.x) < _winSize.x/_pixelLockedPPU) {
				newPosition.x = levelBounds.min.x + (levelBounds.max.x - levelBounds.min.x)/2.0f;
			}

			//clamp min y
			if (newPosition.y < levelBounds.min.y + _winSize.y/_pixelLockedPPU/2.0f) {
				newPosition.y = levelBounds.min.y + _winSize.y/_pixelLockedPPU/2.0f;
			}
			//clamp max y
			else if (newPosition.y > levelBounds.max.y - _winSize.y/_pixelLockedPPU/2.0f) {
				newPosition.y = levelBounds.max.y - _winSize.y/_pixelLockedPPU/2.0f;
			}
			//center y if level height is smaller than camera height
			if (Mathf.Abs(levelBounds.max.y - levelBounds.min.y) < _winSize.y/_pixelLockedPPU) {
				newPosition.y = levelBounds.min.y + (levelBounds.max.y - levelBounds.min.y)/2.0f;
			}

			float nextX = Mathf.Round(_pixelLockedPPU * newPosition.x);
			float nextY = Mathf.Round(_pixelLockedPPU * newPosition.y);
			_camera.transform.position = new Vector3(nextX/_pixelLockedPPU, nextY/_pixelLockedPPU, _camera.transform.position.z);
		}
	}
}
